
/* -*-C-*-
 ##############################################################################
 #
 # File:        trice/src/data.c
 # RCS:         "@(#)$Revision: 1.16 $ $Date: 94/03/09 10:38:16 $"
 # Description: Routines for configuring data format and port parameters 
 #		on a E1430 module.
 # Author:      Doug Passey
 # Created:     
 # Language:    C
 # Package:     E1430
 # Status:      "@(#)$State: Exp $"
 #
 # (C) Copyright 1992, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # Please add additional comments here
 #
 # Revisions:
 #
 ##############################################################################
*/

#    include <stdio.h>

#include "trice.h"
#include "err1430.h"


#ifndef lint
const char data_fileId[] = "$Header: data.c,v 1.16 94/03/09 10:38:16 chriss Exp $";
#endif

/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with the append status 
 * mode <append>.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_append_status_bits(SHORTSIZ16 append, SHORTSIZ16 *bitsPtr)
{
  switch(append) {
    case E1430_APPEND_STATUS_ON:
      *bitsPtr = DATA_FORMAT_APPEND_STATUS_ON;
      break;
    case E1430_APPEND_STATUS_OFF:
      *bitsPtr = DATA_FORMAT_APPEND_STATUS_OFF;
      break;
    default:
      return (i1430_Error(ERR1430_ILLEGAL_APPEND_STATUS, NULL, NULL));
  }

  return (0);
}

/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with <blocksize>. 
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_blocksize_bits(SHORTSIZ16 groupID, LONGSIZ32 blocksize,
							SHORTSIZ16 *bitsPtr)
{
  SHORTSIZ16 i;
  SHORTSIZ16 error;
  LONGSIZ32 size, maxBlockSize;
  SHORTSIZ16 pointSize;

  error = i1430_get_group_data_pt_size(groupID, &pointSize);
  if(error) return(error);

  error = e1430_get_fifo_max_blocksize(groupID, &maxBlockSize);
  if(error) return (error);

  if(blocksize > maxBlockSize) {
    return(i1430_Error(ERR1430_ILLEGAL_BLOCKSIZE, NULL, NULL));
  }else{
    size = blocksize * (LONGSIZ32)pointSize;
  }

  if(size < 8L) { 
    return(i1430_Error(ERR1430_ILLEGAL_BLOCKSIZE, NULL, NULL));
  }

  for(i=-3; i<32; i++) {	/* figure out N, where size = 8*2^N */
    size = size >> 1;
    if(size == 0) break;
  }

  *bitsPtr = i << TRIGGER_BLOCKSIZE_BIT_SHIFT;
  
  return (0);
}

/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with the data type 
 * <type>.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_type_bits(SHORTSIZ16 type, SHORTSIZ16 *bitsPtr)
{
  switch(type) {
    case E1430_DATA_TYPE_REAL:
      *bitsPtr = DATA_FORMAT_DATA_TYPE_REAL;
      break;
    case E1430_DATA_TYPE_COMPLEX:
      *bitsPtr = DATA_FORMAT_DATA_TYPE_COMPLEX;
      break;
    default:
       return(i1430_Error(ERR1430_ILLEGAL_DATA_TYPE, NULL, NULL));
  }

  return (0);
}


/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with the data size 
 * <size>.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_size_bits(SHORTSIZ16 size, SHORTSIZ16 *bitsPtr)
{
  switch(size) {
    case E1430_DATA_SIZE_16:
      *bitsPtr = DATA_FORMAT_DATA_SIZE_16;  
      break;
    case E1430_DATA_SIZE_32:
      *bitsPtr = DATA_FORMAT_DATA_SIZE_32;  
      break;
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_SIZE, NULL, NULL));
  }

  return (0);
}

/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with the reread state
 * <reread>.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_reread_bits(SHORTSIZ16 reread, SHORTSIZ16 *bitsPtr)
{
  switch(reread) {
    case E1430_DATA_REREAD_ON:
      *bitsPtr = TRIGGER_BLOCKSIZE_REREAD_ON;  
      break;
    case E1430_DATA_REREAD_OFF:
      *bitsPtr = TRIGGER_BLOCKSIZE_REREAD_OFF;  
      break;
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_REREAD, NULL, NULL));
  }

  return (0);
}



/*****************************************************************************
 *
 * Return in <bitsPtr> the register bits associated with the data size 
 * <size>.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_mode_bits(SHORTSIZ16 mode, SHORTSIZ16 *bitsPtr)
{
  switch(mode) {
    case E1430_BLOCK_MODE:
      *bitsPtr = DATA_FORMAT_BLOCK_MODE_ON;  
      break;
    case E1430_CONTINUOUS_MODE:
      *bitsPtr = DATA_FORMAT_BLOCK_MODE_OFF;  
      break;
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_MODE, NULL, NULL));
  }

  return (0);
}


/*****************************************************************************
 *
 * Sets up a module group <groupID> to transfer data via either the
 * local bus or the VME bus depending on <port>.
 * <size>.  If more than one module in the group in the LBUS mode, 
 * then all of them must be in contiguous slots on backplane. 
 *
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_port(SHORTSIZ16 groupID, SHORTSIZ16 port)
{
  SHORTSIZ16 la, reg, enable;
  SHORTSIZ16 error;
  aModGroupPtr ptr;

  if((ptr = i1430_valid_module_group(groupID)) == NULL) {/*no such group exists*/
    return (i1430_Error(ERR1430_NO_GROUP, NULL, NULL));
  }

  for(; *ptr != -1; ptr++) {
    la = e1430_modStates[*ptr].logicalAddr;

    error = e1430_read_register_image(la, E1430_HP_PORT_CONTROL_REG, &reg);
    if(error) return(error);
  
    enable = reg & ~PORT_CONTROL_LBUS_ENABLE_MASK;  /* save enable bit */

    /* mask off enable and port control */
    reg &= PORT_CONTROL_LBUS_ENABLE_MASK & PORT_CONTROL_SEND_PORT_MASK;

    switch(port) {
      case E1430_SEND_PORT_VME:   /* VME bus used for data */
        reg |= PORT_CONTROL_LBUS_DISABLE | PORT_CONTROL_SEND_PORT_VME;
        break;

      case E1430_SEND_PORT_LBUS:   /* LBUS */
        reg |= PORT_CONTROL_LBUS_DISABLE | PORT_CONTROL_SEND_PORT_LBUS;
        break;

      default:
        return(i1430_Error(ERR1430_ILLEGAL_DATA_PORT, NULL, NULL));
    }

    error = e1430_write_register_image(la, E1430_HP_PORT_CONTROL_REG, reg);
    if(error) return(error);

    reg &= PORT_CONTROL_LBUS_ENABLE_MASK;  /* mask off enable bit */
    reg |= enable;		/* restore enable bit */

    error = e1430_write_register_image(la, E1430_HP_PORT_CONTROL_REG, reg);
    if(error) return(error);
  }

  return (0);
}

/*****************************************************************************
 *
 * Sets up a module <la> to an LBUS transfer mode.
 *
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_lbus_mode(SHORTSIZ16 la, SHORTSIZ16 mode)
{
  SHORTSIZ16 reg, enable;
  SHORTSIZ16 error;
  char buf[80];

  error = e1430_read_register_image(la, E1430_HP_PORT_CONTROL_REG, &reg);
  if(error) return(error);
  
  enable = reg & ~PORT_CONTROL_LBUS_ENABLE_MASK;

  reg &= (PORT_CONTROL_LBUS_MODE_MASK & PORT_CONTROL_LBUS_ENABLE_MASK);
  reg |= PORT_CONTROL_LBUS_DISABLE;

  switch(mode) {
    case E1430_LBUS_GENERATE:
      reg |= PORT_CONTROL_LBUS_GENERATE;
      break;
    case E1430_LBUS_APPEND:
      reg |= PORT_CONTROL_LBUS_APPEND;
      break;
    case E1430_LBUS_INSERT:
      reg |= PORT_CONTROL_LBUS_INSERT;
      break;
    case E1430_LBUS_PIPELINE:
      reg |= PORT_CONTROL_LBUS_PIPELINE;
      break;
    
    default:
      (void)sprintf(buf, "%hd", (LONGSIZ32)mode);
      return(i1430_Error(ERR1430_ILLEGAL_LBUS_MODE, buf, NULL));
  }

  error = e1430_write_register_image(la, E1430_HP_PORT_CONTROL_REG, reg);
  if(error) return(error);

  reg &= PORT_CONTROL_LBUS_ENABLE_MASK;
  reg |= enable;
  error = e1430_write_register_image(la, E1430_HP_PORT_CONTROL_REG, reg);
  return (error);
}

/*****************************************************************************
 *
 * reset trigger delay bits to reflect change in size of a data point in fifo
 *
 ****************************************************************************/
static SHORTSIZ16 reset_trig_delay(SHORTSIZ16 groupID)
{
  aModGroupPtr ptr;

  ptr = i1430_valid_module_group(groupID);
  return(e1430_set_trigger_delay(groupID, e1430_modStates[*ptr].trigDelaySave));
}


/*****************************************************************************
 * User callable routine to set data format parameters: <type> is REAL or
 * COMPLEX data, <precision> is 16 or 32 bits, and <blocksize> will be set
 * to 8*2^N, chosing N so that the blocksize set is >= to the blocksize 
 * requested. Blocksize of CONTINUOUS_DATA (-1L), will set the module in
 * continuous transfer mode.
 * Returns 0 if all ok, else return negative error number.
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_format(SHORTSIZ16 groupID, SHORTSIZ16 type, 
      SHORTSIZ16 size, SHORTSIZ16 mode, LONGSIZ32 blocksize, SHORTSIZ16 append)
{
  SHORTSIZ16 modSetupBits, bits;
  SHORTSIZ16 error;
  SHORTSIZ16 mask;

  /* set bits for data format */
  error = i1430_get_data_type_bits(type, &bits);
  if(error) return (error);
  modSetupBits = bits;

  /* set bits for data size */
  error = i1430_get_data_size_bits(size, &bits);
  if(error) return (error);
  modSetupBits |= bits;

  /* set bits for appending status */
  error = i1430_get_append_status_bits(append, &bits);
  if(error) return (error);
  modSetupBits |= bits;

  /* get bits for data mode */
  error = i1430_get_data_mode_bits(mode, &bits);
  if(error) return (error);
  modSetupBits |= bits;
       
  mask = DATA_FORMAT_BLOCK_MODE_MASK & DATA_FORMAT_DATA_TYPE_MASK & 
	DATA_FORMAT_DATA_SIZE_MASK & DATA_FORMAT_APPEND_STATUS_MASK;

  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_DATA_FORMAT_REG,
					mask, modSetupBits);
  if(error) return(error);

  /* set bits for blocksize */
  error = i1430_get_blocksize_bits(groupID, blocksize, &bits);
  if(error) return (error);

  modSetupBits = bits;
  mask = TRIGGER_BLOCKSIZE_BLOCK_MASK;

  error = i1430_update_group_module_bits(groupID, E1430_TRIGGER_BLOCKSIZE_REG,
					mask, modSetupBits);
  if(error) return(error);

  /* set register trigger delay bits to reflect new data size or type */
  error = reset_trig_delay(groupID);
  if(error) return(error);
  
  return (i1430_continue_meas(groupID));

}

/*****************************************************************************
 *
 * Set data gathering state, <dataMode>, of module group, <groupID>,
 * to either BLOCK or CONTINUOUS.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_mode(SHORTSIZ16 groupID, SHORTSIZ16 dataMode)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;

  error = i1430_get_data_mode_bits(dataMode, &bits);
  if(error) return (error);
       
  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_DATA_FORMAT_REG, 
				DATA_FORMAT_BLOCK_MODE_MASK, bits);
  if(error) return(error);

  return (i1430_continue_meas(groupID));
}


/*****************************************************************************
 *
 * Set data type, <type>, of module group, <groupID>,
 * to either DATA_TYPE_REAL or DATA_TYPE_COMPLEX.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_type(SHORTSIZ16 groupID, SHORTSIZ16 type)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;
  LONGSIZ32 blocksize;

  /* get current blocksize before data type is changed */
  error = e1430_get_blocksize(groupID, &blocksize);
  if(error) return (error);
       
  error = i1430_get_data_type_bits(type, &bits);
  if(error) return (error);
       
  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_DATA_FORMAT_REG, 
				DATA_FORMAT_DATA_TYPE_MASK, bits);
  if(error) return (error);
 
  /* set register blocksize bits to reflect new data type */
  error = i1430_get_blocksize_bits(groupID, blocksize, &bits);
  if(error) return (error);
       
  error = i1430_update_group_module_bits(groupID, E1430_TRIGGER_BLOCKSIZE_REG, 
				TRIGGER_BLOCKSIZE_BLOCK_MASK, bits);
  if(error) return(error);

  /* set register trigger delay bits to reflect new data type */
  error = reset_trig_delay(groupID);
  if(error) return(error);
  
  return (i1430_continue_meas(groupID));
}


/*****************************************************************************
 *
 * Set data size, <size>, of module group, <groupID> 
 * to either DATA_SIZE_16 or DATA_SIZE_32
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_size(SHORTSIZ16 groupID, SHORTSIZ16 size)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;
  LONGSIZ32 blocksize;

  /* get current blocksize before data size is changed */
  error = e1430_get_blocksize(groupID, &blocksize);
  if(error) return (error);
       
  error = i1430_get_data_size_bits(size, &bits);
  if(error) return (error);
       
  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_DATA_FORMAT_REG, 
				DATA_FORMAT_DATA_SIZE_MASK, bits);
  if(error) return (error);
  
  /* set register blocksize bits to reflect new data type */
  error = i1430_get_blocksize_bits(groupID, blocksize, &bits);
  if(error) return (error);
       
  error = i1430_update_group_module_bits(groupID, E1430_TRIGGER_BLOCKSIZE_REG, 
				TRIGGER_BLOCKSIZE_BLOCK_MASK, bits);
  if(error) return(error);

  /* set register trigger delay bits to reflect new data size */
  error = reset_trig_delay(groupID);
  if(error) return(error);
  
  return (i1430_continue_meas(groupID));
}


/*****************************************************************************
 *
 * Set the append status state, <status>, of module group, <groupID> 
 * to either APPEND_STATUS_ON or APPEND_STATUS_OFF.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_append_status(SHORTSIZ16 groupID, SHORTSIZ16 status)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;

  error = i1430_get_append_status_bits(status, &bits);
  if(error) return (error);
       
  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_DATA_FORMAT_REG, 
				DATA_FORMAT_APPEND_STATUS_MASK, bits);
  if(error) return(error);

  return (i1430_continue_meas(groupID));
}


/*****************************************************************************
 *
 * Set blocksize, <size>, of module group, <groupID>. 
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_blocksize(SHORTSIZ16 groupID, LONGSIZ32 blocksize)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;

  error = i1430_get_blocksize_bits(groupID, blocksize, &bits);
  if(error) return (error);
       
  error =  i1430_pause_meas(groupID);
  if(error) return(error);

  error = i1430_update_group_module_bits(groupID, E1430_TRIGGER_BLOCKSIZE_REG, 
				TRIGGER_BLOCKSIZE_BLOCK_MASK, bits);
  if(error) return(error);

  return (i1430_continue_meas(groupID));
}


/*****************************************************************************
 *
 * Return data gathering mode of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_mode_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].dataFormat & ~DATA_FORMAT_BLOCK_MODE_MASK) {
    case DATA_FORMAT_BLOCK_MODE_ON:
      return (E1430_BLOCK_MODE);
    case DATA_FORMAT_BLOCK_MODE_OFF:
      return(E1430_CONTINUOUS_MODE);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_MODE, NULL, NULL));
  }
}


/*****************************************************************************
 *
 * Returns the data mode of the module group, <groupID>, into
 * the variable pointed to by <modePtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_data_mode(SHORTSIZ16 groupID, SHORTSIZ16 *modePtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_data_mode_index, 
				"data mode", modePtr));
}
 

/*****************************************************************************
 *
 * Return blocksize of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
LONGSIZ32 i1430_get_blocksize_index(SHORTSIZ16 index)
{
  SHORTSIZ16 n;
  SHORTSIZ16 error;
  LONGSIZ32 size;
  SHORTSIZ16 pointSize;

  error = e1430_get_fifo_data_point_size(e1430_modStates[index].logicalAddr, 
								&pointSize);
  if(error) return (error);

  n = (e1430_modStates[index].trigBlocksize & ~TRIGGER_BLOCKSIZE_BLOCK_MASK) >>
					TRIGGER_BLOCKSIZE_BIT_SHIFT;

  size = 1L << (n+3);
  return (size/(LONGSIZ32)pointSize);
}
  

/*****************************************************************************
 *
 * Returns the blocksize of the module group, <groupID>, into
 * the variable pointed to by <blockPtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_blocksize(SHORTSIZ16 groupID, LONGSIZ32 *blockPtr)
{ 
  return (i1430_get_long_parm(groupID, i1430_get_blocksize_index, 
				"blocksize", blockPtr));
}
 

/*****************************************************************************
 *
 * Return data port of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_port_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].portControl & ~PORT_CONTROL_SEND_PORT_MASK) {
    case PORT_CONTROL_SEND_PORT_VME:
      return (E1430_SEND_PORT_VME);
    case PORT_CONTROL_SEND_PORT_LBUS:
      return (E1430_SEND_PORT_LBUS);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_PORT, NULL, NULL));
  }

}


/*****************************************************************************
 *
 * Returns the data port of the module group, <groupID>, into
 * the variable pointed to by <portPtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_data_port(SHORTSIZ16 groupID, SHORTSIZ16 *portPtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_data_port_index, 
				"data port", portPtr));
}
 

/*****************************************************************************
 *
 * Returns the lbus mode of the module, <la>, into
 * the variable pointed to by <modePtr>.  
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_lbus_mode(SHORTSIZ16 la, SHORTSIZ16 *modePtr)
{
  SHORTSIZ16 error, reg;

  *modePtr = 0;

  error = e1430_read_register_image(la, E1430_HP_PORT_CONTROL_REG, &reg);
  if(error) return(error);

  switch(reg & ~ PORT_CONTROL_LBUS_MODE_MASK) {
    case PORT_CONTROL_LBUS_GENERATE:
      *modePtr = E1430_LBUS_GENERATE;
      break;
    case PORT_CONTROL_LBUS_APPEND:
      *modePtr = E1430_LBUS_APPEND;
      break;
    case PORT_CONTROL_LBUS_INSERT:
      *modePtr = E1430_LBUS_INSERT;
      break;
    case PORT_CONTROL_LBUS_PIPELINE:
      *modePtr = E1430_LBUS_PIPELINE;
      break;
  }

  return(0);
}
 

/*****************************************************************************
 *
 * Return append status state of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_append_status_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].dataFormat & ~DATA_FORMAT_APPEND_STATUS_MASK) {
    case DATA_FORMAT_APPEND_STATUS_OFF:
      return (E1430_APPEND_STATUS_OFF);
    case DATA_FORMAT_APPEND_STATUS_ON:
      return (E1430_APPEND_STATUS_ON);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_APPEND_STATUS, NULL, NULL));
  }
}


/*****************************************************************************
 *
 * Returns the append status of the module group, <groupID>, into
 * the variable pointed to by <typePtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_append_status(SHORTSIZ16 groupID, SHORTSIZ16 *statusPtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_append_status_index, 
				"append status", statusPtr));
}
 

/*****************************************************************************
 *
 * Return data type of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_type_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].dataFormat & ~DATA_FORMAT_DATA_TYPE_MASK) {
    case DATA_FORMAT_DATA_TYPE_REAL:
      return (E1430_DATA_TYPE_REAL);
    case DATA_FORMAT_DATA_TYPE_COMPLEX:
      return (E1430_DATA_TYPE_COMPLEX);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_TYPE, NULL, NULL));
  }
}


/*****************************************************************************
 *
 * Returns the data type of the module group, <groupID>, into
 * the variable pointed to by <typePtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_data_type(SHORTSIZ16 groupID, SHORTSIZ16 *typePtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_data_type_index, 
				"data type", typePtr));
}
 

/*****************************************************************************
 *
 * Return data size of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_size_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].dataFormat & ~DATA_FORMAT_DATA_SIZE_MASK) {
    case DATA_FORMAT_DATA_SIZE_16:
      return (E1430_DATA_SIZE_16);
    case DATA_FORMAT_DATA_SIZE_32:
      return (E1430_DATA_SIZE_32);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_SIZE, NULL, NULL));
  }
}


/*****************************************************************************
 *
 * Returns the data size of the module group, <groupID>, into
 * the variable pointed to by <sizePtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_data_size(SHORTSIZ16 groupID, SHORTSIZ16 *sizePtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_data_size_index, 
				"data size", sizePtr));
}
 

/*****************************************************************************
 *
 * Return data reread state of module at <index> in the e1430_modStates array.
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 i1430_get_data_reread_index(SHORTSIZ16 index)
{
  switch(e1430_modStates[index].trigBlocksize & ~TRIGGER_BLOCKSIZE_REREAD_MASK) {
    case TRIGGER_BLOCKSIZE_REREAD_ON:
      return (E1430_DATA_REREAD_ON);
    case TRIGGER_BLOCKSIZE_REREAD_OFF:
      return (E1430_DATA_REREAD_OFF);
    default:
      return (i1430_Error(ERR1430_ILLEGAL_DATA_REREAD, NULL, NULL));
  }
}


/*****************************************************************************
 *
 * Returns the data size of the module group, <groupID>, into
 * the variable pointed to by <rereadPtr>.  
 * Returns 0 if OK, returns ERR1430_PARAMETER_UNEQUAL if the source is
 * not the same for all modules in the group.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_get_data_reread(SHORTSIZ16 groupID, SHORTSIZ16 *rereadPtr)
{ 
  return (i1430_get_short_parm(groupID, i1430_get_data_reread_index, 
				"data reread", rereadPtr));
}
 

/*****************************************************************************
 *
 * Resets or releases from reset the LBUS chip of all the E1430's in <groupID>
 * depending on <state>.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_reset_lbus(SHORTSIZ16 groupID, SHORTSIZ16 state)
{
  SHORTSIZ16 error, reg;
  aModGroupPtr ptr;

  if((ptr = i1430_valid_module_group(groupID)) == NULL) {/*no such group exists*/
    return (i1430_Error(ERR1430_NO_GROUP, NULL, NULL));
  }

  for(; *ptr != -1; ptr++) {
    error = e1430_read_register_image(e1430_modStates[*ptr].logicalAddr,
		E1430_HP_PORT_CONTROL_REG, &reg);
    if(error) return(error);

    reg &= PORT_CONTROL_LBUS_ENABLE_MASK;

    switch(state) {
      case E1430_RESET_LBUS_ON:
	reg |= PORT_CONTROL_LBUS_DISABLE;
	break;

      case E1430_RESET_LBUS_OFF:
	reg |= PORT_CONTROL_LBUS_ENABLE;
	break;

      default:
	return(i1430_Error(ERR1430_ILLEGAL_LBUS_RESET, NULL, NULL));
    }

    error = e1430_write_register_image(e1430_modStates[*ptr].logicalAddr,
		E1430_HP_PORT_CONTROL_REG, reg);
    if(error) return(error);
  }

  return(0);
}
 

/*****************************************************************************
 *
 * Set the data reread state, 
 * Returns 0 if all ok, else return negative error number.
 *
 ****************************************************************************/
SHORTSIZ16 e1430_set_data_reread(SHORTSIZ16 groupID, SHORTSIZ16 reread)
{
  SHORTSIZ16 bits;
  SHORTSIZ16 error;

  error = i1430_get_data_reread_bits(reread, &bits);
  if(error) return (error);
       
  error = i1430_update_group_module_bits(groupID, E1430_TRIGGER_BLOCKSIZE_REG, 
				TRIGGER_BLOCKSIZE_REREAD_MASK, bits);

  return (error);
}


